
#include "onewire110.h"
#include "delay.h"
/*---------------------------------------------------------------------
  Function    : onewire_init
  Description : initialize 1-wire configuration
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void gxt110_onewire_init(void)
{
	
	GPIO_InitTypeDef pin_cfg;
  // enable peripheral clock
  RCC_APB2PeriphClockCmd(ONEWIRE_CLK, ENABLE);
	
  // configure I/O port of DQ
  pin_cfg.GPIO_Pin   = ONEWIRE_PIN;
  pin_cfg.GPIO_Mode  = GPIO_Mode_Out_OD;	  // open-drain output
  pin_cfg.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(ONEWIRE_PORT, &pin_cfg);
	
	GPIO_SetBits(ONEWIRE_PORT, ONEWIRE_PIN);  // default HIGH
}

/*---------------------------------------------------------------------
  Function    : onewire_pulldown            
  Description : pull down the 1-wire bus
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void gxt110_onewire_pulldown(void)
{
  GPIO_ResetBits(ONEWIRE_PORT, ONEWIRE_PIN);
}

/*---------------------------------------------------------------------
  Function    : onewire_release            
  Description : release the 1-wire bus
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void gxt110_onewire_release(void)
{
  GPIO_SetBits(ONEWIRE_PORT, ONEWIRE_PIN);
}

/*---------------------------------------------------------------------
  Function    : onewire_receive
  Description : receive 1 bit of data from the bus
  Parameter   : none
  Return      : data received
---------------------------------------------------------------------*/
uint8_t gxt110_onewire_receive(void)
{
	return GPIO_ReadInputDataBit(ONEWIRE_PORT, ONEWIRE_PIN);
}

/*---------------------------------------------------------------------
  Function    : onewire_reset             
  Description : transmit reset pulse
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void gxt110_onewire_reset(void)
{
	gxt110_onewire_pulldown();
	delay_ms(2);
	gxt110_onewire_release();
    delay_us(100);
}

/*---------------------------------------------------------------------
  Function    : onewire_activate             
  Description : transmit activate pulse
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void gxt110_onewire_activate(void)
{
	gxt110_onewire_pulldown();
	delay_us(500);
	gxt110_onewire_release();
    delay_us(100);
}

/*---------------------------------------------------------------------
  Function    : onewire_read_bit             
  Description : read one bit from the 1-wire bus
  Parameter   : none
  Return      : received bit
---------------------------------------------------------------------*/
uint8_t gxt110_onewire_read_bit(void)
{
  uint8_t rxd = 0;
  
	gxt110_onewire_pulldown();	
	delay_us(8);	// tRL
	gxt110_onewire_release();
	delay_us(7);	// tMSR - tRL
	rxd = gxt110_onewire_receive();
	delay_us(85);	// tSLOT - tMSR
    return rxd;
}

/*---------------------------------------------------------------------
  Function    : onewire_write_bit             
  Description : write one bit to the 1-wire bus
  Parameter   : bit to be transmitted
  Return      : none
---------------------------------------------------------------------*/
void gxt110_onewire_write_bit(uint8_t txd)
{
  if(txd == 0) {
		gxt110_onewire_pulldown();
		delay_us(70);	// tW0L
		gxt110_onewire_release();
		delay_us(30);	// tSLOT - tW0L
	}
  else {
		gxt110_onewire_pulldown();
		delay_us(8);	// tW1L
		gxt110_onewire_release();
		delay_us(92);	// tSLOT - tW1L
	}
}

/*---------------------------------------------------------------------
  Function    : onewire_write_token             
  Description : write token to the 1-wire bus
  Parameter   : token to be transmitted
  Return      : none
---------------------------------------------------------------------*/
void gxt110_onewire_write_token(uint8_t tkn)
{
  uint8_t txd = 0;
  uint8_t i;
  for(i = 0; i < 8; i ++) {
    txd = ((tkn & 0x80) == 0) ? 0x00 : 0x01; // MSB first
    tkn = tkn << 1;
    gxt110_onewire_write_bit(txd);
  }
}
