
#include "gx1833e_driver.h"
#include "usart.h"
#include "delay.h"

#include "gx1833e.h"
#include "usart.h"
/*---------------------------------------------------------------------
  Function    : gx1833e_search
  Description : execute search algorithm
  Parameter   : RID array of devices found
  Return      : number of devices found (0=no device was found)
---------------------------------------------------------------------*/
uint8_t gx1833e_search (uint64_t * ids)
{
  uint8_t  num     = 0;   // number of devices found
  uint8_t  pos     = 0;   // current bit position
  uint8_t  dir     = 0;   // search direction
  uint8_t  val     = 0;   // current bit
  uint8_t  val_cmp = 0;   // complement of current bit
  uint8_t  brn_new = 0;   // the bit position where a new branch is taken
  uint8_t  brn_chk = 0;   // branch checkpoint
  uint64_t cur_rid = 0;   // current RID
  uint64_t reg_rid = 0;   // last RID found
  
  do {
    if(gx1833e_onewire_reset() == GX1833e_ACK) {
      brn_new = 0;
      cur_rid = 0;
      gx1833e_onewire_write_byte(GX1833E_SEARCH);
      for(pos = 1; pos < 65; pos ++) {
        cur_rid = cur_rid >> 1;
        val     = gx1833e_onewire_read_bit();
        val_cmp = gx1833e_onewire_read_bit();
        /*
          00 : There are both 0s and 1s in the current bit position of the participating RIDs => discrepancy
          01 : There are only 0s        in the current bit position of the participating RIDs
          10 : There are only 1s        in the current bit position of the participating RIDs
          11 : No device participating in the search (atypical situation)
        */
        if((val == 0) && (val_cmp == 0)) {
          /*
            pos < brn_chk : take the same path as last time (from last RID found)
            pos = brn_chk : take the  "1" path
            pos > brn_chk : take the  "0" path
          */
          if(pos < brn_chk)
            dir = ((uint8_t) (reg_rid >> (pos - 1))) & 0x01;
          else if(pos == brn_chk)
            dir = 1;
          else
            dir = 0;
          
          if(dir == 0)
            brn_new = pos;
        }
        else if((val == 0) && (val_cmp != 0))
          dir = 0;
        else if((val != 0) && (val_cmp == 0))
          dir = 1;
        else {
          return 0; // Error : the device discovered is removed from the 1-wire bus during the search
        }
        cur_rid = cur_rid | ((dir == 0) ? 0x0000000000000000 : 0x8000000000000000);
        gx1833e_onewire_write_bit(dir);
      }
      brn_chk  = brn_new;
      reg_rid  = cur_rid;
      ids[num] = cur_rid;
      num ++;
    }
    else {
      return 0; // Error : no device on the bus
    }
  } while (brn_chk > 0);
  
  return num; // Returning zero means that no device was found
}

/*---------------------------------------------------------------------
  Function    : gx1833e_alarm
  Description : execute search algorithm (only alarm devices participate)
  Parameter   : RID array of devices found
  Return      : number of devices found (0=no device was found)
---------------------------------------------------------------------*/
uint8_t gx1833e_alarm (uint64_t * ids)
{
  uint8_t  num     = 0;   // number of device found
  uint8_t  pos     = 0;   // current bit position
  uint8_t  dir     = 0;   // search direction
  uint8_t  val     = 0;   // current bit
  uint8_t  val_cmp = 0;   // complement of current bit
  uint8_t  brn_new = 0;   // the bit position where a new branch is taken
  uint8_t  brn_chk = 0;   // branch checkpoint
  uint64_t cur_rid = 0;   // current RID
  uint64_t reg_rid = 0;   // last RID found
  
  do {
    if(gx1833e_onewire_reset() == GX1833e_ACK) {
      brn_new = 0;
      cur_rid = 0;
      gx1833e_onewire_write_byte(GX1833E_ALARM);
      for(pos = 1; pos < 65; pos ++) {
        cur_rid = cur_rid >> 1;
        val     = gx1833e_onewire_read_bit();
        val_cmp = gx1833e_onewire_read_bit();
        /*
          00 : There are both 0s and 1s in the current bit position of the participating RIDs => discrepancy
          01 : There are only 0s        in the current bit position of the participating RIDs
          10 : There are only 1s        in the current bit position of the participating RIDs
          11 : No device participating in the search (atypical situation)
        */
        if((val == 0) && (val_cmp == 0)) {
          /*
            pos < brn_chk : take the same path as last time (from last RID found)
            pos = brn_chk : take the  "1" path
            pos > brn_chk : take the  "0" path
          */
          if(pos < brn_chk)
            dir = ((uint8_t) (reg_rid >> (pos - 1))) & 0x01;
          else if(pos == brn_chk)
            dir = 1;
          else
            dir = 0;
          
          if(dir == 0)
            brn_new = pos;
        }
        else if((val == 0) && (val_cmp != 0))
          dir = 0;
        else if((val != 0) && (val_cmp == 0))
          dir = 1;
        else {
					return 0; // Error : the device discovered is removed from the 1-wire bus during the search
        }
        cur_rid = cur_rid | ((dir == 0) ? 0x0000000000000000 : 0x8000000000000000);
        gx1833e_onewire_write_bit(dir);
      }
      brn_chk  = brn_new;
      reg_rid  = cur_rid;
      ids[num] = cur_rid;
      num ++;
    }
    else {
      return 0; // Error : no device on the bus
    }
  } while (brn_chk > 0);
  
  return num; // Returning zero means that no device was found
}

/*---------------------------------------------------------------------
  Function    : gx1833e_read_rom
  Description : read RID
  Parameter   : none
  Return      : RID of current device (0=no device on the bus)
---------------------------------------------------------------------*/
uint64_t gx1833e_read_rom (void)
{
  uint8_t  rxd = 0;
  uint64_t rid = 0;
  uint8_t i;
  if(gx1833e_onewire_reset() == GX1833e_ACK) {
		gx1833e_onewire_write_byte(GX1833E_READ);
    for( i = 0; i < 8; i ++) {
      rxd = gx1833e_onewire_read_byte();
      rid = rid | (((uint64_t) rxd) << (8 * i));
    }
		return rid;
  }
  else {
		return 0; // Error : no device on the bus
	}
}

/*---------------------------------------------------------------------
  Function    : gx1833e_match
  Description : match RID
  Parameter   : RID of selected device
  Return      : finish flag (0=no device on the bus)
---------------------------------------------------------------------*/
uint8_t gx1833e_match (uint64_t rid)
{
  uint8_t txd = 0;
  uint8_t i;
  if(gx1833e_onewire_reset() == GX1833e_ACK) {
    gx1833e_onewire_write_byte(GX1833E_MATCH);
    for( i = 0; i < 8; i ++) {
      txd = (uint8_t) (rid >> (8 * i));
      gx1833e_onewire_write_byte(txd);
    }
    return 1;
  }
  else {
    return 0; // Error : no device on the bus
  }
}

/*---------------------------------------------------------------------
  Function    : gx1833e_skip
  Description : skip addressing
  Parameter   : none
  Return      : finish flag (0=no device on the bus)
---------------------------------------------------------------------*/
uint8_t gx1833e_skip (void)
{
  if(gx1833e_onewire_reset() == GX1833e_ACK) {
    gx1833e_onewire_write_byte(GX1833E_SKIP);
    return 1;
  }
  else {
    return 0; // Error : no device on the bus
  }
}

/*---------------------------------------------------------------------
  Function    : gx1833e_convert
  Description : start temperature conversion
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void gx1833e_convert (void)
{
  gx1833e_onewire_write_byte(GX1833E_CONVERT);
}


/*---------------------------------------------------------------------
  Function    : gx1833e_search_uid
  Description : execute search algorithm
  Parameter   : UID array of devices found
  Return      : number of devices found (0=no device was found)
---------------------------------------------------------------------*/
uint8_t gx1833e_search_uid (uint32_t * ids)
{
  uint8_t  num     = 0;   // number of devices found
  uint8_t  pos     = 0;   // current bit position
  uint8_t  dir     = 0;   // search direction
  uint8_t  val     = 0;   // current bit
  uint8_t  val_cmp = 0;   // complement of current bit
  uint8_t  brn_new = 0;   // the bit position where a new branch is taken
  uint8_t  brn_chk = 0;   // branch checkpoint
  uint64_t cur_rid = 0;   // current RID
  uint64_t reg_rid = 0;   // last RID found
  
  do {
    if(gx1833e_onewire_reset() == GX1833e_ACK) {
      brn_new = 0;
      cur_rid = 0;
      gx1833e_onewire_write_byte(GX1833E_SEARCH_UID);
      for(pos = 1; pos < 25; pos ++) {
        cur_rid = cur_rid >> 1;
        val     = gx1833e_onewire_read_bit();
        val_cmp = gx1833e_onewire_read_bit();
        /*
          00 : There are both 0s and 1s in the current bit position of the participating RIDs => discrepancy
          01 : There are only 0s        in the current bit position of the participating RIDs
          10 : There are only 1s        in the current bit position of the participating RIDs
          11 : No device participating in the search (atypical situation)
        */
        if((val == 0) && (val_cmp == 0)) {
          /*
            pos < brn_chk : take the same path as last time (from last RID found)
            pos = brn_chk : take the  "1" path
            pos > brn_chk : take the  "0" path
          */
          if(pos < brn_chk)
            dir = ((uint8_t) (reg_rid >> (pos - 1))) & 0x01;
          else if(pos == brn_chk)
            dir = 1;
          else
            dir = 0;
          
          if(dir == 0)
            brn_new = pos;
        }
        else if((val == 0) && (val_cmp != 0))
          dir = 0;
        else if((val != 0) && (val_cmp == 0))
          dir = 1;
        else {
          return 0; // Error : the device discovered is removed from the 1-wire bus during the search
        }
        cur_rid = cur_rid | ((dir == 0) ? 0x00000000 : 0x00800000);
        gx1833e_onewire_write_bit(dir);
      }
      brn_chk  = brn_new;
      reg_rid  = cur_rid;
      ids[num] = cur_rid;
      num ++;
    }
    else {
      return 0; // Error : no device on the bus
    }
  } while (brn_chk > 0);
  
  return num; // Returning zero means that no device was found
}

/*---------------------------------------------------------------------
  Function    : gx1833e_alarm_search_uid
  Description : execute search algorithm (only alarm devices participate)
  Parameter   : RID array of devices found
  Return      : number of devices found (0=no device was found)
---------------------------------------------------------------------*/
uint8_t gx1833e_alarm_search_uid (uint32_t * ids)
{
  uint8_t  num     = 0;   // number of device found
  uint8_t  pos     = 0;   // current bit position
  uint8_t  dir     = 0;   // search direction
  uint8_t  val     = 0;   // current bit
  uint8_t  val_cmp = 0;   // complement of current bit
  uint8_t  brn_new = 0;   // the bit position where a new branch is taken
  uint8_t  brn_chk = 0;   // branch checkpoint
  uint64_t cur_rid = 0;   // current RID
  uint64_t reg_rid = 0;   // last RID found
  
  do {
    if(gx1833e_onewire_reset() == GX1833e_ACK) {
      brn_new = 0;
      cur_rid = 0;
      gx1833e_onewire_write_byte(GX1833E_ALARM_SEARCH_UID);
      for(pos = 1; pos < 25; pos ++) {
        cur_rid = cur_rid >> 1;
        val     = gx1833e_onewire_read_bit();
        val_cmp = gx1833e_onewire_read_bit();
        /*
          00 : There are both 0s and 1s in the current bit position of the participating RIDs => discrepancy
          01 : There are only 0s        in the current bit position of the participating RIDs
          10 : There are only 1s        in the current bit position of the participating RIDs
          11 : No device participating in the search (atypical situation)
        */
        if((val == 0) && (val_cmp == 0)) {
          /*
            pos < brn_chk : take the same path as last time (from last RID found)
            pos = brn_chk : take the  "1" path
            pos > brn_chk : take the  "0" path
          */
          if(pos < brn_chk)
            dir = ((uint8_t) (reg_rid >> (pos - 1))) & 0x01;
          else if(pos == brn_chk)
            dir = 1;
          else
            dir = 0;
          
          if(dir == 0)
            brn_new = pos;
        }
        else if((val == 0) && (val_cmp != 0))
          dir = 0;
        else if((val != 0) && (val_cmp == 0))
          dir = 1;
        else {
					return 0; // Error : the device discovered is removed from the 1-wire bus during the search
        }
        cur_rid = cur_rid | ((dir == 0) ? 0x00000000 : 0x00800000);
        gx1833e_onewire_write_bit(dir);
      }
      brn_chk  = brn_new;
      reg_rid  = cur_rid;
      ids[num] = cur_rid;
      num ++;
    }
    else {
      return 0; // Error : no device on the bus
    }
  } while (brn_chk > 0);
  
  return num; // Returning zero means that no device was found
}

/*---------------------------------------------------------------------
  Function    : gx1833e_match_uid
  Description : match UID
  Parameter   : UID of selected device
  Return      : finish flag (0=no device on the bus)
---------------------------------------------------------------------*/
uint8_t gx1833e_match_uid (uint32_t uid)
{
  uint8_t txd = 0;
  uint8_t i;
  if(gx1833e_onewire_reset() == GX1833e_ACK) {
    gx1833e_onewire_write_byte(GX1833E_MATCH_UID);
    for( i = 0; i < 3; i ++) {
      txd = (uint8_t) (uid >> (8 * i));
      gx1833e_onewire_write_byte(txd);
    }
    return 1;
  }
  else {
    return 0; // Error : no device on the bus
  }
}

/*---------------------------------------------------------------------
  Function    : gx1833e_generate_crc8             
  Description : generate CRC-8 of 1-wire packet
  Parameter   : data packet
  Return      : calculated CRC-8
---------------------------------------------------------------------*/
uint8_t gx1833e_generate_crc8 (uint64_t dat, uint8_t crc_reg_init)
{
  uint8_t dat_in  = 0; // data input bit
  uint8_t crc_in  = 0; // crc  input bit
  uint8_t crc_reg = 0; // crc  register
  uint8_t i;
    
  crc_reg =  crc_reg_init;
    
  for( i = 0; i < 64; i ++) {
    dat_in  = (uint8_t) ((dat >> i) & 0x0000000000000001); // LSB first
    crc_in  = (crc_reg & 0x01) ^ dat_in;
    crc_reg = (crc_reg >> 1) ^ ((crc_in != 0) ? CRC_POLY : 0x00);
  }
  
  return crc_reg;
}




/*---------------------------------------------------------------------
  Function    : gx1833e_read_scratchpad
  Description : read scratchpad
  Parameter   : none
Return      : received data (WrtProt OffsetH OffsetL CONF,TLOW,THIG,TEMP-MSByte,TEMP-LSByte)
---------------------------------------------------------------------*/
uint64_t gx1833e_read_scratchpad (void)
{
	uint8_t  cnt     = 0;
	uint64_t rxd     = 0;
	uint8_t  crc_rxd = 0;
	uint8_t  crc_cal = 0;
	
  gx1833e_onewire_write_byte(GX1833E_READ_SCRATCHPAD);
	
	for(cnt = 0; cnt < 8; cnt ++) {
		rxd = rxd | (((uint64_t) gx1833e_onewire_read_byte()) << (8 * cnt));
	}
	crc_rxd = gx1833e_onewire_read_byte();
	crc_cal = gx1833e_generate_crc8(rxd,0);
	
	if(crc_rxd == crc_cal) {
    return rxd;
	}
	else {
    return 0; // Error : CRC-8 verification failed
	}
}



/*---------------------------------------------------------------------
  Function    : gx1833e_Read_EE
  Description : gx1833e_Read_EE
  Parameter   : none
---------------------------------------------------------------------*/
void gx1833e_Read_EE  (uint8_t addr, uint8_t block)
{
	uint8_t  cnt     = 0;
	uint64_t rxd     = 0;
	uint8_t  crc_rxd = 0;
	uint8_t  crc_cal = 0;
	
   gx1833e_onewire_write_byte(GX1833E_Read_Memory);
   gx1833e_onewire_write_byte(addr);	
   
   while(block--)
   {    
       delay_ms(5);       
        for(cnt = 0; cnt < 8; cnt ++) {
            rxd = rxd | (((uint64_t) gx1833e_onewire_read_byte()) << (8 * cnt));
        }
        crc_rxd = gx1833e_onewire_read_byte();
        crc_cal = gx1833e_generate_crc8(rxd,0);
        
        if(crc_rxd == crc_cal) {
          UartOutDW((rxd>>32));
          UartOutDW(rxd);
          rxd  = 0;
        }
        else {
          UartOutDW(0xEEEEEEEE);
          UartOutDW(0xEEEEEEEE);        
        }
   }
}



const uint64_t prg_array[32]={
    0x0000000000000088,
    0x0000000000008800,
    0x0000000000880000,
    0x0000000088000000,
    0x0000008800000000,
    0x0000880000000000,
    0x0088000000000000,
    0x8800000000000000, 
    0x0000000000000044,
    0x0000000000004400,
    0x0000000000440000,
    0x0000000044000000,
    0x0000004400000000,
    0x0000440000000000,
    0x0044000000000000,
    0x4400000000000000, 
    0x0000000000000022,
    0x0000000000002200,
    0x0000000000220000,
    0x0000000022000000,
    0x0000002200000000,
    0x0000220000000000,
    0x0022000000000000,
    0x2200000000000000, 
    0x0000000000000011,
    0x0000000000001100,
    0x0000000000110000,
    0x0000000011000000,
    0x0000001100000000,
    0x0000110000000000,
    0x0011000000000000,
    0x1100000000000000    

};

/*---------------------------------------------------------------------
  Function    : gx1833e_Program_EE
  Description : gx1833e_Program_EE
  Parameter   : none
---------------------------------------------------------------------*/
void gx1833e_Program_EE  (uint8_t addr, uint8_t block, uint64_t *array)
{
	uint8_t  i     = 0;    
	uint8_t  cnt     = 0;
	uint8_t  crc_rxd = 0;
	uint8_t  crc_cal = 0;
	
   gx1833e_onewire_write_byte(GX1833E_Write_Memory);
   
   for(i = 0; i < block; i ++) 
   {       
       gx1833e_onewire_write_byte(addr);	
       for(cnt = 0; cnt < 8; cnt ++) {
           gx1833e_onewire_write_byte((array[i]>>(cnt*8)));

            
        }

        crc_rxd = gx1833e_onewire_read_byte();
        crc_cal = gx1833e_generate_crc8((uint64_t)addr<<56,0);
        crc_cal = gx1833e_generate_crc8(array[i], crc_cal);
        
        if(crc_rxd == crc_cal) {
        gx1833e_onewire_write_byte(0xA5);
        delay_ms(150);
        addr+=8;
        }
        else {
          UartOutDW(0xEEEEEEEE);
          UartOutDW(0xEEEEEEEE);  
         
        }          
   }
}



/*---------------------------------------------------------------------
  Function    : gx1833e_Program_EE
  Description : gx1833e_Program_EE
  Parameter   : none
---------------------------------------------------------------------*/
void gx1833e_Program_EE_block  (uint8_t addr, uint64_t data)
{
	uint8_t  cnt     = 0;
	uint8_t  crc_rxd = 0;
	uint8_t  crc_cal = 0;
	
   gx1833e_onewire_write_byte(GX1833E_Write_Memory);
       
       gx1833e_onewire_write_byte(addr);	
       for(cnt = 0; cnt < 8; cnt ++) {
           gx1833e_onewire_write_byte((data>>(cnt*8)));
            
        }
        crc_rxd = gx1833e_onewire_read_byte();
        crc_cal = gx1833e_generate_crc8((uint64_t)addr<<56,0);
        crc_cal = gx1833e_generate_crc8(data, crc_cal);
        
        if(crc_rxd == crc_cal) {
        gx1833e_onewire_write_byte(0xA5);
        delay_ms(150);
        }
        else {
          UartOutDW(0xEEEEEEEE);
          UartOutDW(0xEEEEEEEE);        
        }          
}












