/*--------------------------------------------------------------------------------------
  Copyright (c) 2023 GXCAS, Inc. All rights reserved

  File Name   : gx1832_gpio.c
  File Type   : source file
  Author      : zhengkl
  Date        : 2023-04-11
  Version     : 0.1
  Description : DQ port control (based on STM32)

  Modification History :
  Date      By       Version    Change Description
  ====================================================================================
  23/04/11  zhengkl  0.1        Original
--------------------------------------------------------------------------------------*/

/*--------------------------------------------------------------------|
|                     !!!!!!!!!! NOTE !!!!!!!!!!                      |
|                                                                     |
|  The DQ port MUST be configured as open-drain output AT ALL TIMES   |
|--------------------------------------------------------------------*/

#include "gx1832.h"

/*---------------------------------------------------------------------
  Function    : gx1832_gpio_configure
  Description : configure the DQ port
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void gx1832_gpio_configure (void)
{
  // enable peripheral clock
  RCC_APB2PeriphClockCmd(GX1832_GPIO_CLK, ENABLE);
	
  // configure I/O port of DQ
  GPIO_InitTypeDef pin_cfg;
  pin_cfg.GPIO_Pin   = GX1832_GPIO_PIN;
  pin_cfg.GPIO_Mode  = GPIO_Mode_Out_OD;	  // open-drain output
  pin_cfg.GPIO_Speed = GPIO_Speed_50MHz;
  GPIO_Init(GX1832_GPIO_PORT, &pin_cfg);
	
	GPIO_SetBits(GX1832_GPIO_PORT, GX1832_GPIO_PIN);  // default HIGH
}

/*---------------------------------------------------------------------
  Function    : gx1832_gpio_pulldown
  Description : pull down the 1-wire bus
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void gx1832_gpio_pulldown (void)
{
  GPIO_ResetBits(GX1832_GPIO_PORT, GX1832_GPIO_PIN);
}

/*---------------------------------------------------------------------
  Function    : gx1832_gpio_release
  Description : release the 1-wire bus
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void gx1832_gpio_release (void)
{
  GPIO_SetBits(GX1832_GPIO_PORT, GX1832_GPIO_PIN);
}

/*---------------------------------------------------------------------
  Function    : gx1832_gpio_sample
  Description : sample the 1-wire bus
  Parameter   : none
  Return      : sampled data
---------------------------------------------------------------------*/
uint8_t gx1832_gpio_sample (void)
{
	return GPIO_ReadInputDataBit(GX1832_GPIO_PORT, GX1832_GPIO_PIN);
}

/*---------------------------------------------------------------------
  Function    : gx1832_gpio_delay
  Description : implement microsecond delay
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void gx1832_gpio_delay (uint16_t us)
{
  /* regSet of SysTick
    CTRL : Control and status register
      1. countFlag (bit-16) : VAL=0 => countFlag=1 (automatically clear when reading this bit)
      2. clkSource (bit-2)  : 0=External clock (AHB/8), 1=Internal clock (AHB)
      3. tickInt   (bit-1)  : 0=No action when zero VAL, 1=Trigger interrupt when zero VAL
      4. enable    (bit-0)  : 0=disable, 1=enable
    LOAD : Reload  Value Register
    VAL  : Current value Register
    CALIB: Calibration   Register
  */
  SysTick->LOAD  = 9 * (us - 1);  // 9 at AHB/8, 72 at AHB
  SysTick->VAL   = 0;
  SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;
  while((SysTick->CTRL & SysTick_CTRL_COUNTFLAG_Msk) == 0);
  SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
}
