/*--------------------------------------------------------------------------------------
  Copyright (c) 2023 GXCAS, Inc. All rights reserved

  File Name   : gx1832_driver.c
  File Type   : source file
  Author      : zhengkl
  Date        : 2023-04-11
  Version     : 0.1
  Description : driver for GX1832

  Modification History :
  Date      By       Version    Change Description
  ====================================================================================
  23/04/11  zhengkl  0.1        Original
--------------------------------------------------------------------------------------*/

#include "gx1832.h"

/*---------------------------------------------------------------------
  Function    : gx1832_search
  Description : execute search algorithm
  Parameter   : RID array of devices found
  Return      : number of devices found (0=no device was found)
---------------------------------------------------------------------*/
uint8_t gx1832_search (uint64_t * ids)
{
  uint8_t  num     = 0;   // number of devices found
  uint8_t  pos     = 0;   // current bit position
  uint8_t  dir     = 0;   // search direction
  uint8_t  val     = 0;   // current bit
  uint8_t  val_cmp = 0;   // complement of current bit
  uint8_t  brn_new = 0;   // the bit position where a new branch is taken
  uint8_t  brn_chk = 0;   // branch checkpoint
  uint64_t cur_rid = 0;   // current RID
  uint64_t reg_rid = 0;   // last RID found
  
  do {
    if(gx1832_onewire_reset() == GX1832_ACK) {
      brn_new = 0;
      cur_rid = 0;
      gx1832_onewire_write_byte(GX1832_SEARCH);
      for(pos = 1; pos < 65; pos ++) {
        cur_rid = cur_rid >> 1;
        val     = gx1832_onewire_read_bit();
        val_cmp = gx1832_onewire_read_bit();
        /*
          00 : There are both 0s and 1s in the current bit position of the participating RIDs => discrepancy
          01 : There are only 0s        in the current bit position of the participating RIDs
          10 : There are only 1s        in the current bit position of the participating RIDs
          11 : No device participating in the search (atypical situation)
        */
        if((val == 0) && (val_cmp == 0)) {
          /*
            pos < brn_chk : take the same path as last time (from last RID found)
            pos = brn_chk : take the  "1" path
            pos > brn_chk : take the  "0" path
          */
          if(pos < brn_chk)
            dir = ((uint8_t) (reg_rid >> (pos - 1))) & 0x01;
          else if(pos == brn_chk)
            dir = 1;
          else
            dir = 0;
          
          if(dir == 0)
            brn_new = pos;
        }
        else if((val == 0) && (val_cmp != 0))
          dir = 0;
        else if((val != 0) && (val_cmp == 0))
          dir = 1;
        else {
          return 0; // Error : the device discovered is removed from the 1-wire bus during the search
        }
        cur_rid = cur_rid | ((dir == 0) ? 0x0000000000000000 : 0x8000000000000000);
        gx1832_onewire_write_bit(dir);
      }
      brn_chk  = brn_new;
      reg_rid  = cur_rid;
      ids[num] = cur_rid;
      num ++;
    }
    else {
      return 0; // Error : no device on the bus
    }
  } while (brn_chk > 0);
  
  return num; // Returning zero means that no device was found
}

/*---------------------------------------------------------------------
  Function    : gx1832_alarm
  Description : execute search algorithm (only alarm devices participate)
  Parameter   : RID array of devices found
  Return      : number of devices found (0=no device was found)
---------------------------------------------------------------------*/
uint8_t gx1832_alarm (uint64_t * ids)
{
  uint8_t  num     = 0;   // number of device found
  uint8_t  pos     = 0;   // current bit position
  uint8_t  dir     = 0;   // search direction
  uint8_t  val     = 0;   // current bit
  uint8_t  val_cmp = 0;   // complement of current bit
  uint8_t  brn_new = 0;   // the bit position where a new branch is taken
  uint8_t  brn_chk = 0;   // branch checkpoint
  uint64_t cur_rid = 0;   // current RID
  uint64_t reg_rid = 0;   // last RID found
  
  do {
    if(gx1832_onewire_reset() == GX1832_ACK) {
      brn_new = 0;
      cur_rid = 0;
      gx1832_onewire_write_byte(GX1832_ALARM);
      for(pos = 1; pos < 65; pos ++) {
        cur_rid = cur_rid >> 1;
        val     = gx1832_onewire_read_bit();
        val_cmp = gx1832_onewire_read_bit();
        /*
          00 : There are both 0s and 1s in the current bit position of the participating RIDs => discrepancy
          01 : There are only 0s        in the current bit position of the participating RIDs
          10 : There are only 1s        in the current bit position of the participating RIDs
          11 : No device participating in the search (atypical situation)
        */
        if((val == 0) && (val_cmp == 0)) {
          /*
            pos < brn_chk : take the same path as last time (from last RID found)
            pos = brn_chk : take the  "1" path
            pos > brn_chk : take the  "0" path
          */
          if(pos < brn_chk)
            dir = ((uint8_t) (reg_rid >> (pos - 1))) & 0x01;
          else if(pos == brn_chk)
            dir = 1;
          else
            dir = 0;
          
          if(dir == 0)
            brn_new = pos;
        }
        else if((val == 0) && (val_cmp != 0))
          dir = 0;
        else if((val != 0) && (val_cmp == 0))
          dir = 1;
        else {
					return 0; // Error : the device discovered is removed from the 1-wire bus during the search
        }
        cur_rid = cur_rid | ((dir == 0) ? 0x0000000000000000 : 0x8000000000000000);
        gx1832_onewire_write_bit(dir);
      }
      brn_chk  = brn_new;
      reg_rid  = cur_rid;
      ids[num] = cur_rid;
      num ++;
    }
    else {
      return 0; // Error : no device on the bus
    }
  } while (brn_chk > 0);
  
  return num; // Returning zero means that no device was found
}

/*---------------------------------------------------------------------
  Function    : gx1832_read
  Description : read RID
  Parameter   : none
  Return      : RID of current device (0=no device on the bus)
---------------------------------------------------------------------*/
uint64_t gx1832_read (void)
{
  uint8_t  rxd = 0;
  uint64_t rid = 0;
  
  if(gx1832_onewire_reset() == GX1832_ACK) {
		gx1832_onewire_write_byte(GX1832_READ);
    for(uint8_t i = 0; i < 8; i ++) {
      rxd = gx1832_onewire_read_byte();
      rid = rid | (((uint64_t) rxd) << (8 * i));
    }
		return rid;
  }
  else {
		return 0; // Error : no device on the bus
	}
}

/*---------------------------------------------------------------------
  Function    : gx1832_match
  Description : match RID
  Parameter   : RID of selected device
  Return      : finish flag (0=no device on the bus)
---------------------------------------------------------------------*/
uint8_t gx1832_match (uint64_t rid)
{
  uint8_t txd = 0;
  
  if(gx1832_onewire_reset() == GX1832_ACK) {
    gx1832_onewire_write_byte(GX1832_MATCH);
    for(uint8_t i = 0; i < 8; i ++) {
      txd = (uint8_t) (rid >> (8 * i));
      gx1832_onewire_write_byte(txd);
    }
    return 1;
  }
  else {
    return 0; // Error : no device on the bus
  }
}

/*---------------------------------------------------------------------
  Function    : gx1832_skip
  Description : skip addressing
  Parameter   : none
  Return      : finish flag (0=no device on the bus)
---------------------------------------------------------------------*/
uint8_t gx1832_skip (void)
{
  if(gx1832_onewire_reset() == GX1832_ACK) {
    gx1832_onewire_write_byte(GX1832_SKIP);
    return 1;
  }
  else {
    return 0; // Error : no device on the bus
  }
}

/*---------------------------------------------------------------------
  Function    : gx1832_convert
  Description : start temperature conversion
  Parameter   : none
  Return      : none
---------------------------------------------------------------------*/
void gx1832_convert (void)
{
  gx1832_onewire_write_byte(GX1832_CONVERT);
}

/*---------------------------------------------------------------------
  Function    : gx1832_read_temperature
  Description : read temperature
  Parameter   : none
  Return      : temperature (signed integers, LSB=0.0625C)
---------------------------------------------------------------------*/
int16_t gx1832_read_temperature (void)
{
  uint8_t cnt = 0;
  int16_t rxd = 0;
  
  gx1832_onewire_write_byte(GX1832_READ_SCRATCHPAD);
  
  for(cnt = 0; cnt < 2; cnt ++) {
    rxd = rxd | (((int16_t) gx1832_onewire_read_byte()) << (8 * cnt));
  }
  
  return rxd;
}

/*---------------------------------------------------------------------
  Function    : gx1832_read_scratchpad
  Description : read scratchpad
  Parameter   : none
  Return      : received data (CONF,TLOW,THIG,TEMP-MSByte,TEMP-LSByte)
---------------------------------------------------------------------*/
uint64_t gx1832_read_scratchpad (void)
{
	uint8_t  cnt     = 0;
	uint64_t rxd     = 0;
	uint8_t  crc_rxd = 0;
	uint8_t  crc_cal = 0;
	
  gx1832_onewire_write_byte(GX1832_READ_SCRATCHPAD);
	
	for(cnt = 0; cnt < 5; cnt ++) {
		rxd = rxd | (((uint64_t) gx1832_onewire_read_byte()) << (8 * cnt));
	}
	crc_rxd = gx1832_onewire_read_byte();
	crc_cal = gx1832_generate_crc8(rxd);
	
	if(crc_rxd == crc_cal) {
    return rxd;
	}
	else {
    return 0; // Error : CRC-8 verification failed
	}
}

/*---------------------------------------------------------------------
  Function    : gx1832_write_scratchpad
  Description : write scratchpad
  Parameter   : data to be write (CONF,TLOW,THIG)
  Return      : none
---------------------------------------------------------------------*/
void gx1832_write_scratchpad (uint64_t txd)
{
	uint8_t cnt = 0;
	
  gx1832_onewire_write_byte(GX1832_WRITE_SCRATCHPAD);
	
	for(cnt = 0; cnt < 3; cnt ++) {
	  gx1832_onewire_write_byte((uint8_t) txd);
		txd = txd >> 8;
	}
}

/*---------------------------------------------------------------------
  Function    : gx1832_generate_crc8             
  Description : generate CRC-8 of 1-wire packet
  Parameter   : data packet
  Return      : calculated CRC-8
---------------------------------------------------------------------*/
uint8_t gx1832_generate_crc8 (uint64_t dat)
{
  uint8_t dat_in  = 0; // data input bit
  uint8_t crc_in  = 0; // crc  input bit
  uint8_t crc_reg = 0; // crc  register
  
  for(uint8_t i = 0; i < 40; i ++) {
    dat_in  = (uint8_t) ((dat >> i) & 0x0000000000000001); // LSB first
    crc_in  = (crc_reg & 0x01) ^ dat_in;
    crc_reg = (crc_reg >> 1) ^ ((crc_in != 0) ? CRC_POLY : 0x00);
  }
  
  return crc_reg;
}
